package fr.jussieu.ufr_info_p7.T9;

/**
 * A class implement the <code>Completion</code> interface to provide
 * completion services to <code>T9</code> class instances.
 *
 * A completion is a two-part word: a prefix read from user inputs, and a
 * completion suffix provided by the completion service itself. Any user
 * input fires
 * events to a {@code T9} object which invoke methods to the completion
 * service. In a <code>T9</code> object, a completion can be viewed as a word
 * (the prefix) and a possible completion word (the suffix). A basic navigation
 * service in provided through <code>completionPrecedente()</code> and
 * <code>completionSuivante()</code> methods. When the user effectively choose
 * to complete the word, the <code>T9</code> object fires
 * the <code>complete()</code> method.
 * <P>A minimal completion service could be:
 * <pre>
 class BasicCompletion implements Completion {
   private StringBuffer prefix = new StringBuffer();
   public void termine() { }
   public void complete() { }
   public void setMotCourant(String mot) {
     if (mot==null) throw new NullPointerException();
     prefix = new StringBuffer(mot);
   }
   public String getMotCourant() { return prefix.toString(); }
   public String getCompletionCourante() { return ""; }
   public void completionPrecedente() { }
   public void completionSuivante() { }
   public void ajouteCaractere(char c) { prefixe.append(c); }
   public void enleveDernierCaractere() {
     if (prefix.length()>0) { prefix.setLength(prefix.length()-1); }
   }
 }
</pre>
 *
 * @author  Jean-Baptiste Yunès
 * @version 1.01, 2004-11-30
 * @see     T9
 */
public interface Completion {
    /**
     * This method is called before the <code>T9</code> object terminates. This
     * slot is primarily provided to clean internal data structures.
     * <P>The
     * general contract is that it is invoked when <code>T9</code> object has
     * decided to terminate with your current <code>completion</code> object.
     * <P>Important note: this method is not devised to terminate the JVM,
     * you are not supposed to call any termination function.
     */
    public void termine();
    /**
     * Set the value of the completion prefix. This method is called
     * when the <code>T9</code> object needs to reset the current completion.
     * A completion protocol conform method must update the completion
     * suffix according
     * to the new value.
     *
     * @param motCourant the new value to use as the prefix.
     * @throws NullPointerException if a <code>null</code> value
     * is used
     */
    public void setMotCourant(String motCourant);
    /**
     * Get the value of the prefix. This method is called whenever the
     * UI needs to show the value.
     *
     * @return the value of the prefix.
     */
    public String getMotCourant();
    /**
     * Get the value of the suffix. This method is called whenever the
     * UI needs to show the value.
     *
     * @return the value of the suffix.
     */
    public String getCompletionCourante();
    /**
     * Complete the prefix with the curreent suffix.
     * This method is called when the user wants to complete the
     * prefix with the current suffix.
     */ 
    public void complete();
    /**
     * Add a character to the prefix. This method is called when the user
     * input a character through the UI. This basically adds the character
     * to the prefix.
     *
     * @param caractere the input character.
     */
    public void ajouteCaractere(char caractere);
    /**
     * Remove the last character of the prefix. This method is called when
     * the user wants to delete the last character of the prefix. This method
     * will never be called if the length of the <code>String</code> returned
     * by <code><a href="#getMotCourant()">getMotCourant()</a></code> is 0.
     *
     */
    public void enleveDernierCaractere();
    /**
     * Compute the previous possible completion of the current prefix.
     * This method
     * is called whenever the <code>T9</code> want to navigate back through
     * the list of possible completion words. The preceding word must be
     * provided through <code><a href="#getCompletionCourante()">getCompletionCourante()</a></code> method.
     */
    public void completionPrecedente();
    /**
     * Compute the next possible completion of the current prefix. This method
     * is called whenever the <code>T9</code> want to navigate forward through
     * the list of possible completion words. The next word must be
     * available through <code><a href="#getCompletionCourante()">getCompletionCourante()</a></code> method.
     */
    public void completionSuivante();
}