package fr.up7.info;

import java.io.InputStream;
import java.io.FilterInputStream;
import java.io.IOException;

/**
 * A <code>RLEInputStream</code> implements a stream filter to read
 * compressed data using a RLE format.
 *
 * The format is such that any too long running subsequence of the same single
 * letter in the original data is encoded as the length of the run and the
 * letter. As <code>aaaabbdeaaa</code> could simply be encoded as 
 * <code>*4abbde*3a</code>.
 * @version 0.1 (10/2005) 
 * @author Jean-Baptiste Yunès
 * @see FilterInputStream
 * @since 1.4
 */
public class RLEInputStream extends FilterInputStream {
    private long count;
    private int rl, rc; // Run-Length & Running-Char
    private boolean running; // Are we running ?
    private boolean closed;
    /**
     * Constructs a decompressing input stream from a given input stream.
     * @param is the underlying compressed data input stream.
     * @throws RLEMissingHeaderException if no header is found in the file
     * @throws RLEUnknownHeaderException if an unknown header is found.
     */
    public RLEInputStream(InputStream is) throws RLEHeaderException {
	super(is);
	count = 0;
	byte [] h = new byte[5];
	try {
	    if (is.read(h,0,h.length)!=h.length)
		throw new RLEMissingHeaderException();
	} catch (IOException e) {
	    throw new RLEMissingHeaderException();
	}
	if (!RLEHeader.isAnHeader(h))
	    throw new RLEUnknownHeaderException();
	count += h.length;
	running = false;
	closed = false;
    }
    /**
     * Actually not supported.
     * @return <code>false</code>
     * @see #mark
     * @see #reset
     */
    public boolean markSupported() {
	return false;
    }
    /**
     * Actually not supported.
     * @see #markSupported
     */
    public void mark(int limit) {
    }
    /**
     * Actually not supported.
     * @throws IOException each time that function is called.
     * @see #markSupported
     */
    public void reset() throws IOException {
	throw new IOException("mark() unsupported");
    }
    /**
     * Reads a byte of uncompressed data.
     * This method blocks until enough input is available for uncompressing
     * more data.
     *
     * @return the next byte of uncompressed data.
     * @throws IOException if an I/O error has occured.
     * @see FilterInputStream#in
     */
    public int read() throws IOException {
	if (closed) throw new IOException();
	if (running) {
	    count++;
	    rl--;
	    running = rl!=0;
	    return rc;
	}
	rc = super.read();
	switch(rc) {
	case RLE.MARKER:
	    rc = super.read();
	    switch(rc) {
	    case -1:
		throw new RLEBadDataException();
	    default:
		rl = rc;
		rc = super.read();
		if (rc==-1) throw new RLEBadDataException();
		running = true;
		rl--;
		count++;
	    case RLE.MARKER:
	    }
	case -1:
	    break;
	default:
	    count++;
	}
	return rc;
    }
    /**
     * Closes the stream.
     * @throws IOException if an I/O error has occured.
     * @see FilterInputStream#in
     */
    public void close() throws IOException {
	closed = true;
	super.close();
    }
    /**
     * Gives the number of uncompressed bytes already read.
     * @return the number of bytes read.
     */
    public long offset() {
	return count;
    }
}