package fr.up7.info;

import java.io.OutputStream;
import java.io.FilterOutputStream;
import java.io.IOException;

/**
 * A <code>RLEOutputStream</code> implements a stream filter to compress data
 * and physically write it using a RLE format.
 *
 * The format is such that any too long running subsequence of the same single
 * letter in the original data is encoded as the length of the run and the
 * letter. As <code>aaaabbdeaaa</code> could simply be encoded as 
 * <code>*4abbde*3a</code>.
 * @version 0.1 (10/2005) 
 * @author Jean-Baptiste Yunès
 * @see FilterOutputStream
 * @since 1.4
 */
public class RLEOutputStream extends FilterOutputStream {
    private int count;
    private int rl, rc; // Run-Length & Running-Char
    private boolean closed;
    /**
     * Constructs a compressing output stream from a given output stream.
     * @param os the underlying output stream.
     * @throws RLEMissingHeaderException if no header is found in the file
     * @throws RLEUnknownHeaderException if an unknown header is found.
     */
     public RLEOutputStream(OutputStream os) throws RLEHeaderException {
	super(os);
	count = 0;
 	try {
 	    byte [] h = RLEHeader.getHeader_0_1();
 	    os.write(h);
	    count += h.length;
 	} catch (IOException e) {
 	    throw new RLEMissingHeaderException();
 	}
	rc = -1;
	rl = 0;
	closed = false;
    }
    /**
     * Writes a byte of uncompressed data.
     *
     * @throws IOException if an I/O error has occured.
     * @see FilterOutputStream#out
     */
    public void write(int b) throws IOException {
	if (closed) throw new IOException();
	if (b==rc && rl<254) {
	    rl++;
	    return;
	}
	internalFlushRL();
	rc = b;
	rl = 1;
    }
    private void internalFlushRL() throws IOException {
	switch(rl) {
	case 3:
	    super.write(rc); count++;
	case 2:
	    super.write(rc); count++;
	case 1:
	    super.write(rc); count++;
	case 0:
	    break;
	default:
	    super.write(RLE.MARKER); count++;
	    super.write(rl); count++;
	    super.write(rc); count++;
	    break;
	}
    }
    /**
     * Closes the stream.
     *
     * @throws IOException if an I/O error has occured.
     * @see FilterOutputStream#out
     */
    public void close() throws IOException {
	flush();
	super.close();
	closed = true;
    }

    /**
     * Flushes this output stream, forces any buffered data to be written
     *
     * @throws IOException if an I/O error has occured
     */
    public void flush() throws IOException {
	internalFlushRL();
	rc = -1;
	rl = 0;
    }
    /**
     * Gets the length of the compressed data stream.
     *
     * @return the length of the data stream
     */
    public int offset() {
	return count;
    }
}